<?php
header('Content-Type: application/json');
include '../../config/db.php';

// 🧩 INPUTS
$day       = isset($_GET['day']) ? intval($_GET['day']) : 0;
$username  = $_GET['username'] ?? null;
$key       = $_GET['key'] ?? null;

// 🔒 Access Key Check
if ($key !== 'access') {
    echo json_encode(['status' => 'error', 'message' => 'No access']);
    exit;
}

// 🕒 Timezone
$connect->query("SET time_zone = '+05:30'");

// 📅 Date Range
$startDate   = date('Y-m-d', strtotime("-$day days"));
$currentDate = date('Y-m-d');

// 🧩 Get Default Commission (from referel table)
$settings = mysqli_fetch_assoc(mysqli_query($connect, "SELECT referel_comisen FROM referel LIMIT 1"));
$default_commission = floatval($settings['referel_comisen'] ?? 0);

// 🧩 Function: Get commission %
function getCommissionPercent($connect, $username, $default_commission)
{
    $stmt = $connect->prepare("SELECT refer_by, custome_ref FROM users2 WHERE username=? LIMIT 1");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $res = $stmt->get_result();
    $user = $res->fetch_assoc();
    $stmt->close();

    if (empty($user['refer_by'])) {
        // No referral → no commission
        return 0;
    }

    if (!empty($user['custome_ref']) && floatval($user['custome_ref']) > 0) {
        // Custom commission exists
        return floatval($user['custome_ref']);
    }

    // Otherwise use default
    return $default_commission;
}

// 🧩 If single username given
if ($username) {
    $commissionPercent = getCommissionPercent($connect, $username, $default_commission);

    $sql = "SELECT 
                SUM(price) AS total_spent,
                SUM(api_charge) AS total_charge
            FROM orders
            WHERE username=? 
            AND DATE(order_create) BETWEEN ? AND ?";
    $stmt = $connect->prepare($sql);
    $stmt->bind_param("sss", $username, $startDate, $currentDate);
    $stmt->execute();
    $res = $stmt->get_result();
    $data = $res->fetch_assoc();
    $stmt->close();

    $total_spent  = floatval($data['total_spent'] ?? 0);
    $total_charge = floatval($data['total_charge'] ?? 0);

    // Apply commission if applicable
    if ($commissionPercent > 0) {
        $total_spent = $total_spent * (100 - $commissionPercent) / 100;
    }
}
// 🧩 Otherwise all users
else {
    $sql = "SELECT 
                u.username,
                u.refer_by,
                u.custome_ref,
                SUM(o.price) AS total_spent,
                SUM(o.api_charge) AS total_charge
            FROM orders o
            INNER JOIN users2 u ON o.username = u.username
            WHERE DATE(o.order_create) BETWEEN ? AND ?
            GROUP BY u.username";

    $stmt = $connect->prepare($sql);
    $stmt->bind_param("ss", $startDate, $currentDate);
    $stmt->execute();
    $res = $stmt->get_result();

    $total_spent = 0;
    $total_charge = 0;

    while ($row = $res->fetch_assoc()) {
        $client_spent = floatval($row['total_spent'] ?? 0);
        $client_charge = floatval($row['total_charge'] ?? 0);

        // Only apply commission if referred
        if (!empty($row['refer_by'])) {
            $commissionPercent = (!empty($row['custome_ref']) && floatval($row['custome_ref']) > 0)
                ? floatval($row['custome_ref'])
                : $default_commission;

            $client_spent = $client_spent * (100 - $commissionPercent) / 100;
        }

        $total_spent += $client_spent;
        $total_charge += $client_charge;
    }

    $stmt->close();
}

// 💰 Profit Calculation
$total_profit = $total_spent - $total_charge;

// 🧾 Final Response
echo json_encode([
    'status' => 'success',
    'day' => $day,
    'from_date' => $startDate,
    'to_date' => $currentDate,
    'username' => $username ?? 'All Users',
    'total_spent' => number_format($total_spent, 2),
    'total_charge' => number_format($total_charge, 2),
    'total_profit' => number_format($total_profit, 2)
]);

$connect->close();
?>
